<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\BankAccount;
use App\Models\DigitalPayment;
use App\Models\ExpenseType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $query = Transaction::where('user_id', auth()->id())
            ->with(['bankAccount', 'digitalPayment', 'expenseType']);

        // Filter by date range
        if ($request->filled('from_date')) {
            $query->whereDate('date', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('date', '<=', $request->to_date);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by payment method
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        $transactions = $query->orderBy('date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(20);

        $totalIncome = $query->clone()->where('type', 'income')->sum('amount');
        $totalExpense = $query->clone()->where('type', 'expense')->sum('amount');

        return view('transactions.index', compact('transactions', 'totalIncome', 'totalExpense'));
    }

    public function create()
    {
        $bankAccounts = auth()->user()->bankAccounts()->where('is_active', true)->get();
        $digitalPayments = auth()->user()->digitalPayments()->where('is_active', true)->get();
        $expenseTypes = ExpenseType::where(function ($query) {
            $query->where('user_id', auth()->id())
                ->orWhere('is_system', true);
        })->orderBy('name')->get();

        return view('transactions.create', compact('bankAccounts', 'digitalPayments', 'expenseTypes'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'type' => 'required|in:income,expense,transfer',
            'payment_method' => 'required|in:cash,bank,digital',
            // Only validate bank account when payment_method is bank
            'bank_account_id' => 'exclude_unless:payment_method,bank|required|exists:bank_accounts,id',
            // Only validate digital payment when payment_method is digital
            'digital_payment_id' => 'exclude_unless:payment_method,digital|required|exists:digital_payments,id',
            // Only validate expense type when type is expense
            'expense_type_id' => 'exclude_unless:type,expense|required|exists:expense_types,id',
            'amount' => 'required|numeric|min:0.01',
            'details' => 'nullable|string|max:500',
            'reference' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:500',
        ]);

        $user = auth()->user();

        DB::transaction(function () use ($validated, $user) {
            // Get current balance based on payment method
            $currentBalance = $this->getCurrentBalance($validated['payment_method'],
                $validated['bank_account_id'] ?? null,
                $validated['digital_payment_id'] ?? null);

            // Calculate new balance
            if ($validated['type'] === 'income') {
                $newBalance = $currentBalance + $validated['amount'];
            } else {
                $newBalance = $currentBalance - $validated['amount'];
            }

            // Create transaction
            $transaction = $user->transactions()->create([
                'date' => $validated['date'],
                'type' => $validated['type'],
                'payment_method' => $validated['payment_method'],
                'bank_account_id' => $validated['bank_account_id'] ?? null,
                'digital_payment_id' => $validated['digital_payment_id'] ?? null,
                'expense_type_id' => $validated['expense_type_id'] ?? null,
                'amount' => $validated['amount'],
                'balance_before' => $currentBalance,
                'balance_after' => $newBalance,
                'details' => $validated['details'],
                'reference' => $validated['reference'],
                'notes' => $validated['notes'],
            ]);

            // Update account balance
            $this->updateAccountBalance($validated['payment_method'],
                $validated['bank_account_id'] ?? null,
                $validated['digital_payment_id'] ?? null,
                $newBalance);

            // Update user total balance
            $this->updateUserBalance($user, $validated['type'], $validated['amount']);
        });

        return redirect()->route('transactions.index')
            ->with('success', 'Transaction created successfully.');
    }

    private function getCurrentBalance($method, $bankId = null, $digitalId = null)
    {
        if ($method === 'cash') {
            $lastTransaction = Transaction::where('user_id', auth()->id())
                ->where('payment_method', 'cash')
                ->orderBy('id', 'desc')
                ->first();
            return $lastTransaction ? $lastTransaction->balance_after : 0;
        }

        if ($method === 'bank' && $bankId) {
            $account = BankAccount::find($bankId);
            return $account ? $account->balance : 0;
        }

        if ($method === 'digital' && $digitalId) {
            $account = DigitalPayment::find($digitalId);
            return $account ? $account->balance : 0;
        }

        return 0;
    }

    private function updateAccountBalance($method, $bankId = null, $digitalId = null, $newBalance)
    {
        if ($method === 'bank' && $bankId) {
            BankAccount::where('id', $bankId)->update(['balance' => $newBalance]);
        }

        if ($method === 'digital' && $digitalId) {
            DigitalPayment::where('id', $digitalId)->update(['balance' => $newBalance]);
        }
    }

    private function updateUserBalance($user, $type, $amount)
    {
        if ($type === 'income') {
            $user->increment('total_balance', $amount);
        } else {
            $user->decrement('total_balance', $amount);
        }
    }

    public function cashIndex(Request $request)
    {
        $query = Transaction::where('user_id', auth()->id())
            ->where('payment_method', 'cash')
            ->with(['bankAccount', 'digitalPayment', 'expenseType']);

        // Filter by date range
        if ($request->filled('from_date')) {
            $query->whereDate('date', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('date', '<=', $request->to_date);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        $transactions = $query->orderBy('date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(20);

        $totalIncome = $query->clone()->where('type', 'income')->sum('amount');
        $totalExpense = $query->clone()->where('type', 'expense')->sum('amount');

        return view('transactions.cash', compact('transactions', 'totalIncome', 'totalExpense'));
    }

    public function show(Transaction $transaction)
    {
        $this->authorize('view', $transaction);
        return view('transactions.show', compact('transaction'));
    }
}
