<?php

namespace App\Http\Controllers;

use App\Models\SalesDashboard;
use App\Models\Product;
use App\Models\Particular;
use App\Models\BankAccount;
use App\Models\DigitalPayment;
use App\Models\ExpenseType;
use App\Models\Segment;
use App\Models\Transaction;
use Illuminate\Http\Request;

class SalesDashboardController extends Controller
{
    public function index()
    {
        $salesDashboards = SalesDashboard::orderBy('created_at', 'desc')->get();
        return view('sales-dashboard.index', compact('salesDashboards'));
    }

public function create()
    {
        $products = Product::orderBy('name', 'asc')->get();
        $particulars = Particular::orderBy('particulars', 'asc')->get();
        $bankAccounts = BankAccount::orderBy('bank_name', 'asc')->get();
        $digitalPayments = DigitalPayment::orderBy('provider', 'asc')->get();
        $expenseTypes = ExpenseType::orderBy('name', 'asc')->get();
        $segments = Segment::orderBy('name', 'asc')->get();
        return view('sales-dashboard.create', compact('products', 'particulars', 'bankAccounts', 'digitalPayments', 'expenseTypes', 'segments'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'date' => 'nullable|date',
            'invoice' => 'nullable|string|max:255',
            'product' => 'nullable|string|max:255',
            'qty' => 'nullable|integer|min:0',
            'sales' => 'nullable|numeric|min:0',
            'total_sales' => 'nullable|numeric',
            'particulars' => 'nullable|string',
            'payment' => 'nullable|numeric|min:0',
            'payment_method' => 'nullable|in:cash,bank,digital',
            'bank_account_id' => 'required_if:payment_method,bank|nullable|exists:bank_accounts,id',
            'digital_payment_id' => 'required_if:payment_method,digital|nullable|exists:digital_payments,id',
            'due' => 'nullable|numeric|min:0',
            'cogs' => 'nullable|numeric|min:0',
            'gross_profit' => 'nullable|numeric',
            'percentage' => 'nullable|numeric',
            'cash_received_by' => 'nullable|string|max:255',
            'segment' => 'nullable|string|max:255',
            'amount' => 'nullable|numeric|min:0',
            'kpi' => 'nullable|numeric|min:0',
            'revenue' => 'nullable|numeric|min:0',
            'achiv' => 'nullable|numeric|min:0',
        ]);

        // Remove fields that shouldn't be stored in database
        unset($validated['total_sales']);

        // Add user_id
        $validated['user_id'] = auth()->id();

        // Process payment if applicable
        if ($validated['payment'] > 0) {
            if ($validated['payment_method'] === 'bank') {
                $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);

                // Add payment amount to bank account
                $balanceBefore = $bankAccount->balance;
                $bankAccount->update(['balance' => $bankAccount->balance + $validated['payment']]);

                // Create transaction record
                Transaction::create([
                    'user_id' => auth()->id(),
                    'date' => $validated['date'] ?? now(),
                    'type' => 'income',
                    'details' => 'Sales Payment: ' . ($validated['invoice'] ?? 'N/A'),
                    'payment_method' => 'bank',
                    'bank_account_id' => $bankAccount->id,
                    'amount' => $validated['payment'],
                    'balance_before' => $balanceBefore,
                    'balance_after' => $bankAccount->balance,
                    'reference' => 'Sales Payment',
                    'notes' => 'Payment received for sale. Invoice: ' . ($validated['invoice'] ?? 'N/A')
                ]);

            } elseif ($validated['payment_method'] === 'digital') {
                $digitalPayment = DigitalPayment::findOrFail($validated['digital_payment_id']);

                // Add payment amount to digital payment account
                $balanceBefore = $digitalPayment->balance;
                $digitalPayment->update(['balance' => $digitalPayment->balance + $validated['payment']]);

                // Create transaction record
                Transaction::create([
                    'user_id' => auth()->id(),
                    'date' => $validated['date'] ?? now(),
                    'type' => 'income',
                    'details' => 'Sales Payment: ' . ($validated['invoice'] ?? 'N/A'),
                    'payment_method' => 'digital',
                    'digital_payment_id' => $digitalPayment->id,
                    'amount' => $validated['payment'],
                    'balance_before' => $balanceBefore,
                    'balance_after' => $digitalPayment->balance,
                    'reference' => 'Sales Payment',
                    'notes' => 'Payment received for sale. Invoice: ' . ($validated['invoice'] ?? 'N/A')
                ]);
            }
        }

        SalesDashboard::create($validated);

        return redirect()->route('sales-dashboard.index')
            ->with('success', 'Sales dashboard record created successfully.');
    }

    public function edit(SalesDashboard $salesDashboard)
    {
        $bankAccounts = BankAccount::orderBy('bank_name', 'asc')->get();
        $digitalPayments = DigitalPayment::orderBy('provider', 'asc')->get();
        return view('sales-dashboard.edit', compact('salesDashboard', 'bankAccounts', 'digitalPayments'));
    }

    public function update(Request $request, SalesDashboard $salesDashboard)
    {
        $validated = $request->validate([
            'date' => 'nullable|date',
            'invoice' => 'nullable|string|max:255',
            'product' => 'nullable|string|max:255',
            'qty' => 'nullable|integer|min:0',
            'sales' => 'nullable|numeric|min:0',
            'total_sales' => 'nullable|numeric',
            'particulars' => 'nullable|string',
            'payment' => 'nullable|numeric|min:0',
            'payment_method' => 'nullable|in:cash,bank,digital',
            'bank_account_id' => 'required_if:payment_method,bank|nullable|exists:bank_accounts,id',
            'digital_payment_id' => 'required_if:payment_method,digital|nullable|exists:digital_payments,id',
            'due' => 'nullable|numeric|min:0',
            'cogs' => 'nullable|numeric|min:0',
            'gross_profit' => 'nullable|numeric',
            'percentage' => 'nullable|numeric',
            'cash_received_by' => 'nullable|string|max:255',
            'segment' => 'nullable|string|max:255',
            'amount' => 'nullable|numeric|min:0',
            'kpi' => 'nullable|numeric|min:0',
            'revenue' => 'nullable|numeric|min:0',
            'achiv' => 'nullable|numeric|min:0',
        ]);

        // Remove fields that shouldn't be stored in database
        unset($validated['total_sales']);

        // Handle payment changes
        $oldPayment = $salesDashboard->payment ?? 0;
        $newPayment = $validated['payment'] ?? 0;
        $paymentDifference = $newPayment - $oldPayment;

        if (abs($paymentDifference) > 0.01) {
            // If payment was previously from an account, reverse that transaction
            if ($oldPayment > 0) {
                if ($salesDashboard->payment_method === 'bank' && $salesDashboard->bank_account_id) {
                    $bankAccount = BankAccount::find($salesDashboard->bank_account_id);
                    if ($bankAccount) {
                        $bankAccount->update(['balance' => $bankAccount->balance - $oldPayment]);
                    }
                } elseif ($salesDashboard->payment_method === 'digital' && $salesDashboard->digital_payment_id) {
                    $digitalPayment = DigitalPayment::find($salesDashboard->digital_payment_id);
                    if ($digitalPayment) {
                        $digitalPayment->update(['balance' => $digitalPayment->balance - $oldPayment]);
                    }
                }
            }

            // Add new payment to the new account if specified
            if ($newPayment > 0) {
                if ($validated['payment_method'] === 'bank') {
                    $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);
                    $balanceBefore = $bankAccount->balance;
                    $bankAccount->update(['balance' => $bankAccount->balance + $newPayment]);

                    Transaction::create([
                        'user_id' => auth()->id(),
                        'date' => $validated['date'] ?? now(),
                        'type' => 'income',
                        'details' => 'Sales Payment Updated: ' . ($validated['invoice'] ?? 'N/A'),
                        'payment_method' => 'bank',
                        'bank_account_id' => $bankAccount->id,
                        'amount' => $newPayment,
                        'balance_before' => $balanceBefore,
                        'balance_after' => $bankAccount->balance,
                        'reference' => 'Sales Payment Updated',
                        'notes' => 'Payment updated for sale. Invoice: ' . ($validated['invoice'] ?? 'N/A')
                    ]);

                } elseif ($validated['payment_method'] === 'digital') {
                    $digitalPayment = DigitalPayment::findOrFail($validated['digital_payment_id']);
                    $balanceBefore = $digitalPayment->balance;
                    $digitalPayment->update(['balance' => $digitalPayment->balance + $newPayment]);

                    Transaction::create([
                        'user_id' => auth()->id(),
                        'date' => $validated['date'] ?? now(),
                        'type' => 'income',
                        'details' => 'Sales Payment Updated: ' . ($validated['invoice'] ?? 'N/A'),
                        'payment_method' => 'digital',
                        'digital_payment_id' => $digitalPayment->id,
                        'amount' => $newPayment,
                        'balance_before' => $balanceBefore,
                        'balance_after' => $digitalPayment->balance,
                        'reference' => 'Sales Payment Updated',
                        'notes' => 'Payment updated for sale. Invoice: ' . ($validated['invoice'] ?? 'N/A')
                    ]);
                }
            }
        }

        $salesDashboard->update($validated);

        return redirect()->route('sales-dashboard.index')
            ->with('success', 'Sales dashboard record updated successfully.');
    }

    public function destroy(SalesDashboard $salesDashboard)
    {
        $salesDashboard->delete();

        return redirect()->route('sales-dashboard.index')
            ->with('success', 'Sales dashboard record deleted successfully.');
    }
}
