<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Invoice;
use App\Models\Expense;
use App\Models\BankAccount;
use App\Models\DigitalPayment;
use App\Models\Segment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function profitLoss(Request $request)
    {
        $user = auth()->user();
        $year = $request->get('year', now()->year);
        $month = $request->get('month', now()->month);

        // Get income for the period
        $income = Transaction::where('user_id', $user->id)
            ->where('type', 'income')
            ->whereYear('date', $year)
            ->whereMonth('date', $month)
            ->sum('amount');

        // Get expenses for the period
        $expenses = Transaction::where('user_id', $user->id)
            ->where('type', 'expense')
            ->whereYear('date', $year)
            ->whereMonth('date', $month)
            ->sum('amount');

        // Get expense breakdown by type
        $expenseBreakdown = Expense::where('user_id', $user->id)
            ->with('expenseType')
            ->whereYear('date', $year)
            ->whereMonth('date', $month)
            ->select('expense_type_id', DB::raw('SUM(amount) as total'))
            ->groupBy('expense_type_id')
            ->get();

        // Get segment-wise expenses
        $segmentExpenses = Expense::where('user_id', $user->id)
            ->with('segment')
            ->whereYear('date', $year)
            ->whereMonth('date', $month)
            ->whereNotNull('segment_id')
            ->select('segment_id', DB::raw('SUM(amount) as total'))
            ->groupBy('segment_id')
            ->get();

        $netProfit = $income - $expenses;

        return view('reports.profit-loss', compact(
            'income', 'expenses', 'netProfit',
            'expenseBreakdown', 'segmentExpenses',
            'year', 'month'
        ));
    }

    public function cashBook(Request $request)
    {
        $user = auth()->user();
        $fromDate = $request->get('from_date', now()->startOfMonth()->toDateString());
        $toDate = $request->get('to_date', now()->toDateString());

        $transactions = Transaction::where('user_id', $user->id)
            ->where('payment_method', 'cash')
            ->whereBetween('date', [$fromDate, $toDate])
            ->orderBy('date', 'asc')
            ->orderBy('id', 'asc')
            ->get();

        $openingBalance = Transaction::where('user_id', $user->id)
            ->where('payment_method', 'cash')
            ->where('date', '<', $fromDate)
            ->orderBy('date', 'desc')
            ->orderBy('id', 'desc')
            ->value('balance_after') ?? 0;

        $totalReceipts = $transactions->where('type', 'income')->sum('amount');
        $totalPayments = $transactions->where('type', 'expense')->sum('amount');
        $closingBalance = $openingBalance + $totalReceipts - $totalPayments;

        return view('reports.cash-book', compact(
            'transactions', 'openingBalance', 'closingBalance',
            'totalReceipts', 'totalPayments', 'fromDate', 'toDate'
        ));
    }

    public function bankBook(Request $request)
    {
        $user = auth()->user();
        $fromDate = $request->get('from_date', now()->startOfMonth()->toDateString());
        $toDate = $request->get('to_date', now()->toDateString());
        $bankAccountId = $request->get('bank_account_id');

        $bankAccounts = $user->bankAccounts()->where('is_active', true)->get();

        $query = Transaction::where('user_id', $user->id)
            ->where('payment_method', 'bank')
            ->whereBetween('date', [$fromDate, $toDate]);

        if ($bankAccountId) {
            $query->where('bank_account_id', $bankAccountId);
        }

        $transactions = $query->with('bankAccount')
            ->orderBy('date', 'asc')
            ->orderBy('id', 'asc')
            ->get();

        $openingBalance = $this->getOpeningBankBalance($user, $bankAccountId, $fromDate);
        $totalDeposits = $transactions->where('type', 'income')->sum('amount');
        $totalWithdrawals = $transactions->where('type', 'expense')->sum('amount');
        $closingBalance = $openingBalance + $totalDeposits - $totalWithdrawals;

        return view('reports.bank-book', compact(
            'transactions', 'bankAccounts', 'bankAccountId',
            'openingBalance', 'closingBalance', 'totalDeposits',
            'totalWithdrawals', 'fromDate', 'toDate'
        ));
    }

    private function getOpeningBankBalance($user, $bankAccountId, $fromDate)
    {
        $query = Transaction::where('user_id', $user->id)
            ->where('payment_method', 'bank')
            ->where('date', '<', $fromDate);

        if ($bankAccountId) {
            $query->where('bank_account_id', $bankAccountId);
        }

        $lastTransaction = $query->orderBy('date', 'desc')
            ->orderBy('id', 'desc')
            ->first();

        if ($lastTransaction) {
            return $lastTransaction->balance_after;
        }

        if ($bankAccountId) {
            $account = BankAccount::find($bankAccountId);
            return $account ? $account->balance : 0;
        }

        return 0;
    }

    public function balanceSheet(Request $request)
    {
        $user = auth()->user();
        $asAt = $request->get('as_at', now()->toDateString());

        // Assets
        $cashBalance = Transaction::where('user_id', $user->id)
            ->where('payment_method', 'cash')
            ->whereDate('date', '<=', $asAt)
            ->orderBy('date', 'desc')
            ->orderBy('id', 'desc')
            ->value('balance_after') ?? 0;

        $bankAccounts = $user->bankAccounts()
            ->where('is_active', true)
            ->get()
            ->map(function ($account) {
                return [
                    'name' => $account->account_name . ' (' . $account->bank_name . ')',
                    'balance' => $account->balance
                ];
            });

        $digitalAccounts = $user->digitalPayments()
            ->where('is_active', true)
            ->get()
            ->map(function ($account) {
                return [
                    'name' => $account->provider . ' - ' . $account->account_number,
                    'balance' => $account->balance
                ];
            });

        $totalBankBalance = $bankAccounts->sum('balance');
        $totalDigitalBalance = $digitalAccounts->sum('balance');

        // Accounts Receivable (unpaid invoices)
        $accountsReceivable = Invoice::where('user_id', $user->id)
            ->where('due', '>', 0)
            ->whereDate('date', '<=', $asAt)
            ->sum('due');

        // Inventory value
        $inventoryValue = $user->products()
            ->select(DB::raw('SUM(current_stock * purchase_price) as total'))
            ->value('total') ?? 0;

        // Liabilities
        $accountsPayable = 0; // Implement if you have purchase on credit

        // Equity
        $retainedEarnings = $this->calculateRetainedEarnings($user, $asAt);

        $totalAssets = $cashBalance + $totalBankBalance + $totalDigitalBalance +
                      $accountsReceivable + $inventoryValue;
        $totalLiabilities = $accountsPayable;
        $totalEquity = $retainedEarnings;
        $totalLiabilitiesAndEquity = $totalLiabilities + $totalEquity;

        return view('reports.balance-sheet', compact(
            'cashBalance', 'bankAccounts', 'digitalAccounts',
            'totalBankBalance', 'totalDigitalBalance',
            'accountsReceivable', 'inventoryValue',
            'accountsPayable', 'retainedEarnings',
            'totalAssets', 'totalLiabilities', 'totalEquity',
            'totalLiabilitiesAndEquity', 'asAt'
        ));
    }

    private function calculateRetainedEarnings($user, $asAt)
    {
        $totalIncome = Transaction::where('user_id', $user->id)
            ->where('type', 'income')
            ->whereDate('date', '<=', $asAt)
            ->sum('amount');

        $totalExpense = Transaction::where('user_id', $user->id)
            ->where('type', 'expense')
            ->whereDate('date', '<=', $asAt)
            ->sum('amount');

        return $totalIncome - $totalExpense;
    }

    public function salesReport(Request $request)
    {
        $user = auth()->user();
        $fromDate = $request->get('from_date', now()->startOfMonth()->toDateString());
        $toDate = $request->get('to_date', now()->toDateString());

        // Get all invoices with items
        $invoices = Invoice::where('user_id', $user->id)
            ->with(['items.product', 'customer'])
            ->whereBetween('date', [$fromDate, $toDate])
            ->orderBy('date', 'asc')
            ->get();

        // Flatten invoice items for detailed sales table
        $salesLines = $invoices->flatMap(function ($invoice) {
            return $invoice->items->map(function ($item) use ($invoice) {
                return [
                    'date' => $invoice->date,
                    'invoice_id' => $invoice->id,
                    'invoice_no' => $invoice->invoice_no,
                    'product_name' => $item->product->name ?? '-',
                    'qty' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'total_sales' => $item->total,
                    'payment_method' => $this->getPaymentMethod($invoice),
                    'paid' => $invoice->paid,
                    'due' => $invoice->due,
                    'cogs' => $item->cogs,
                    'gross_profit' => $item->profit ?? ($item->total - $item->cogs),
                    'percentage' => $item->total > 0 ? (($item->profit ?? ($item->total - $item->cogs)) / $item->total * 100) : 0,
                    'customer_name' => $invoice->customer->name ?? '-'
                ];
            });
        });

        // Summary calculations
        $summary = [
            'total_invoices' => $invoices->count(),
            'total_sales' => $invoices->sum('total'),
            'total_paid' => $invoices->sum('paid'),
            'total_due' => $invoices->sum('due'),
            'total_cogs' => $invoices->sum(function ($invoice) {
                return $invoice->items->sum(function ($item) {
                    return $item->cogs ?? 0;
                });
            }),
            'total_profit' => $invoices->sum(function ($invoice) {
                return $invoice->items->sum(function ($item) {
                    return $item->profit ?? ($item->total - ($item->cogs ?? 0));
                });
            }),
        ];

        $summary['profit_margin'] = $summary['total_sales'] > 0
            ? ($summary['total_profit'] / $summary['total_sales'] * 100)
            : 0;

        // Payment method breakdown
        $paymentBreakdown = $invoices->groupBy(function ($invoice) {
            return $this->getPaymentMethod($invoice);
        })->map(function ($group) {
            return [
                'total' => $group->sum('paid'),
                'count' => $group->count()
            ];
        });

        // Day-wise sales breakdown
        $dayWiseSales = $invoices->groupBy(function ($invoice) {
            return $invoice->date->day;
        })->map(function ($dayInvoices, $day) {
            $totalQty = $dayInvoices->sum(function ($invoice) {
                return $invoice->items->sum('quantity');
            });
            $totalAmount = $dayInvoices->sum('total');
            $totalCogs = $dayInvoices->sum(function ($invoice) {
                return $invoice->items->sum(function ($item) {
                    return $item->cogs ?? 0;
                });
            });
            $costPerItem = $totalQty > 0 ? ($totalCogs / $totalQty) : 0;

            return [
                'day' => $day,
                'qty' => $totalQty,
                'amount' => $totalAmount,
                'cogs' => $totalCogs,
                'cost_per_item' => $costPerItem
            ];
        })->sortBy('day');

        // Segment-wise sales
        $segmentWiseSales = Segment::all()->map(function ($segment) use ($invoices) {
            // For now, segment sales calculation - you may need to add segment_id to invoice if not present
            $segmentInvoices = $invoices; // Placeholder calculation

            return [
                'segment_name' => $segment->name,
                'qty' => 0,
                'amount' => 0,
                'kpi' => 0,
                'revenue' => 0,
                'achievement' => '0%'
            ];
        });
        // Daily sales chart data
        $dailySales = $invoices->groupBy(function ($invoice) {
            return $invoice->date->toDateString();
        })->map(function ($group, $date) {
            return [
                'date' => $date,
                'total' => $group->sum('total'),
                'count' => $group->count()
            ];
        })->values();

        return view('reports.sales', compact(
            'invoices', 'salesLines', 'summary', 'dailySales',
            'paymentBreakdown', 'dayWiseSales', 'segmentWiseSales',
            'fromDate', 'toDate'
        ));
    }

    private function getPaymentMethod($invoice)
    {
        // Determine payment method based on paid amount
        if ($invoice->paid == 0) {
            return 'Pending';
        } elseif ($invoice->due == 0) {
            return 'Paid';
        } else {
            return 'Partial';
        }
    }

    public function receivablePayable(Request $request)
    {
        $user = auth()->user();
        $asAt = $request->get('as_at', now()->toDateString());

        // Get all unpaid/partial invoices (receivables)
        $receivables = Invoice::where('user_id', $user->id)
            ->whereIn('status', ['pending', 'partial'])
            ->where('due', '>', 0)
            ->with('customer')
            ->orderBy('due_date', 'asc')
            ->get();

        $totalReceivable = $receivables->sum('due');

        // For now, payables are not implemented
        $totalPayable = 0;
        $payables = collect([]);

        return view('reports.receivable-payable', compact(
            'receivables', 'payables',
            'totalReceivable', 'totalPayable',
            'asAt'
        ));
    }
}
