<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\BankAccount;
use App\Models\DigitalPayment;
use App\Models\Transaction;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::where('user_id', auth()->id())
            ->orderBy('name')
            ->get();

        return view('products.index', compact('products'));
    }

    public function create()
    {
        $bankAccounts = BankAccount::where('user_id', auth()->id())->where('is_active', true)->get();
        $digitalPayments = DigitalPayment::where('user_id', auth()->id())->where('is_active', true)->get();

        $product = new Product();
        return view('products.create', compact('bankAccounts', 'digitalPayments', 'product'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100|unique:products,sku,NULL,id,user_id,' . auth()->id(),
            'description' => 'nullable|string',
            'unit' => 'nullable|string|max:50',
            'current_stock' => 'nullable|integer|min:0',
            'minimum_stock' => 'nullable|integer|min:0',
            'purchase_price' => 'nullable|numeric|min:0',
            'sales_price' => 'nullable|numeric|min:0',
            'payment_method' => 'required|in:bank,digital',
            'bank_account_id' => 'required_if:payment_method,bank|nullable|exists:bank_accounts,id',
            'digital_payment_id' => 'required_if:payment_method,digital|nullable|exists:digital_payments,id',
        ]);

        // Calculate total purchase cost
        $totalCost = ($validated['purchase_price'] ?? 0) * ($validated['current_stock'] ?? 0);

        if ($totalCost > 0) {
            // Deduct money from selected payment method
            if ($validated['payment_method'] === 'bank') {
                $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);

                if ($bankAccount->balance < $totalCost) {
                    return back()->withInput()->with('error', 'Insufficient balance in selected bank account. Available: ' . $bankAccount->balance . ', Required: ' . $totalCost);
                }

                // Deduct balance
                $bankAccount->update(['balance' => $bankAccount->balance - $totalCost]);

                // Create transaction record
                Transaction::create([
                    'user_id' => auth()->id(),
                    'date' => now(),
                    'type' => 'expense',
                    'details' => 'Product Purchase: ' . $validated['name'],
                    'payment_method' => 'bank',
                    'bank_account_id' => $bankAccount->id,
                    'amount' => $totalCost,
                    'balance_before' => $bankAccount->balance + $totalCost,
                    'balance_after' => $bankAccount->balance,
                    'reference' => 'Product Purchase',
                    'notes' => 'Purchased ' . ($validated['current_stock'] ?? 0) . ' units at ' . ($validated['purchase_price'] ?? 0) . ' each'
                ]);

            } elseif ($validated['payment_method'] === 'digital') {
                $digitalPayment = DigitalPayment::findOrFail($validated['digital_payment_id']);

                if ($digitalPayment->balance < $totalCost) {
                    return back()->withInput()->with('error', 'Insufficient balance in selected digital payment. Available: ' . $digitalPayment->balance . ', Required: ' . $totalCost);
                }

                // Deduct balance
                $digitalPayment->update(['balance' => $digitalPayment->balance - $totalCost]);

                // Create transaction record
                Transaction::create([
                    'user_id' => auth()->id(),
                    'date' => now(),
                    'type' => 'expense',
                    'details' => 'Product Purchase: ' . $validated['name'],
                    'payment_method' => 'digital',
                    'digital_payment_id' => $digitalPayment->id,
                    'amount' => $totalCost,
                    'balance_before' => $digitalPayment->balance + $totalCost,
                    'balance_after' => $digitalPayment->balance,
                    'reference' => 'Product Purchase',
                    'notes' => 'Purchased ' . ($validated['current_stock'] ?? 0) . ' units at ' . ($validated['purchase_price'] ?? 0) . ' each'
                ]);
            }
        }

        auth()->user()->products()->create($validated);

        return redirect()->route('products.index')
            ->with('success', 'Product created successfully and payment processed.');
    }

    public function edit(Product $product)
    {
        $this->authorize('update', $product);
        return view('products.edit', compact('product'));
    }

    public function update(Request $request, Product $product)
    {
        $this->authorize('update', $product);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100|unique:products,sku,' . $product->id . ',id,user_id,' . auth()->id(),
            'description' => 'nullable|string',
            'unit' => 'nullable|string|max:50',
            'current_stock' => 'nullable|integer|min:0',
            'minimum_stock' => 'nullable|integer|min:0',
            'purchase_price' => 'nullable|numeric|min:0',
            'sales_price' => 'nullable|numeric|min:0',
        ]);

        $product->update($validated);

        return redirect()->route('products.index')
            ->with('success', 'Product updated successfully.');
    }

    public function destroy(Product $product)
    {
        $this->authorize('delete', $product);

        if ($product->invoiceItems()->exists() || $product->stockMovements()->exists()) {
            return back()->with('error', 'Cannot delete product with associated records.');
        }

        $product->delete();

        return redirect()->route('products.index')
            ->with('success', 'Product deleted successfully.');
    }

    public function stockIndex()
    {
        $products = Product::where('user_id', auth()->id())
            ->whereColumn('current_stock', '<=', 'minimum_stock')
            ->get();

        return view('products.stock', compact('products'));
    }

    public function adjustStock(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer',
            'type' => 'required|in:add,remove',
            'notes' => 'nullable|string',
        ]);

        $product = Product::where('id', $validated['product_id'])
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $quantity = $validated['type'] === 'add'
            ? $product->current_stock + abs($validated['quantity'])
            : max(0, $product->current_stock - abs($validated['quantity']));

        $product->update(['current_stock' => $quantity]);

        // Create stock movement record
        $product->stockMovements()->create([
            'type' => $validated['type'],
            'quantity' => abs($validated['quantity']),
            'notes' => $validated['notes'],
            'user_id' => auth()->id(),
        ]);

        return redirect()->route('products.index')
            ->with('success', 'Stock adjusted successfully.');
    }
}
