<?php

namespace App\Http\Controllers;

use App\Models\DigitalPayment;
use Illuminate\Http\Request;

class DigitalPaymentController extends Controller
{
    public function index()
    {
        $accounts = auth()->user()->digitalPayments()
            ->with('transactions')
            ->orderBy('is_active', 'desc')
            ->orderBy('provider')
            ->get();

        return view('digital-payments.index', compact('accounts'));
    }

    public function create()
    {
        return view('digital-payments.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'provider' => 'required|string|max:255',
            'account_number' => 'nullable|string|max:50',
            'account_name' => 'required|string|max:255',
            'initial_balance' => 'nullable|numeric|min:0',
        ]);

        $account = auth()->user()->digitalPayments()->create([
            'provider' => $validated['provider'],
            'account_number' => $validated['account_number'],
            'account_name' => $validated['account_name'],
            'balance' => $validated['initial_balance'] ?? 0,
            'is_active' => true,
        ]);

        if ($validated['initial_balance'] > 0) {
            $account->user->transactions()->create([
                'date' => now(),
                'type' => 'income',
                'payment_method' => 'digital',
                'digital_payment_id' => $account->id,
                'amount' => $validated['initial_balance'],
                'balance_before' => 0,
                'balance_after' => $validated['initial_balance'],
                'details' => 'Initial balance for ' . $account->provider . ' - ' . $account->account_name,
                'reference' => 'INITIAL',
            ]);
        }

        return redirect()->route('digital-payments.index')
            ->with('success', 'Digital payment account created successfully.');
    }

    public function edit(DigitalPayment $digitalPayment)
    {
        $this->authorize('update', $digitalPayment);
        return view('digital-payments.edit', compact('digitalPayment'));
    }

    public function update(Request $request, DigitalPayment $digitalPayment)
    {
        $this->authorize('update', $digitalPayment);

        $validated = $request->validate([
            'provider' => 'required|string|max:255',
            'account_number' => 'nullable|string|max:50',
            'account_name' => 'required|string|max:255',
            'is_active' => 'boolean',
        ]);

        $digitalPayment->update($validated);

        return redirect()->route('digital-payments.index')
            ->with('success', 'Digital payment account updated successfully.');
    }

    public function destroy(DigitalPayment $digitalPayment)
    {
        $this->authorize('delete', $digitalPayment);

        if ($digitalPayment->transactions()->exists()) {
            return back()->with('error', 'Cannot delete account with transactions.');
        }

        $digitalPayment->delete();

        return redirect()->route('digital-payments.index')
            ->with('success', 'Digital payment account deleted successfully.');
    }
}
