<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Invoice;
use App\Models\Expense;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();

        // Get today's transactions
        $todayIncome = Transaction::where('user_id', $user->id)
            ->where('type', 'income')
            ->whereDate('date', today())
            ->sum('amount');

        $todayExpense = Transaction::where('user_id', $user->id)
            ->where('type', 'expense')
            ->whereDate('date', today())
            ->sum('amount');

        // Get monthly summary
        $monthlyIncome = Transaction::where('user_id', $user->id)
            ->where('type', 'income')
            ->whereMonth('date', now()->month)
            ->whereYear('date', now()->year)
            ->sum('amount');

        $monthlyExpense = Transaction::where('user_id', $user->id)
            ->where('type', 'expense')
            ->whereMonth('date', now()->month)
            ->whereYear('date', now()->year)
            ->sum('amount');

        // Get account balances
        $cashBalance = Transaction::where('user_id', $user->id)
            ->where('payment_method', 'cash')
            ->orderBy('id', 'desc')
            ->value('balance_after') ?? 0;

        $bankAccounts = $user->bankAccounts()->where('is_active', true)->get();
        $digitalAccounts = $user->digitalPayments()->where('is_active', true)->get();

        // Get recent transactions
        $recentTransactions = Transaction::where('user_id', $user->id)
            ->with(['bankAccount', 'digitalPayment', 'expenseType'])
            ->orderBy('date', 'desc')
            ->limit(10)
            ->get();

        // Get low stock products
        $lowStockProducts = Product::where('user_id', $user->id)
            ->whereColumn('current_stock', '<=', 'minimum_stock')
            ->limit(5)
            ->get();

        // Chart data for last 30 days
        $chartData = $this->getChartData($user);

        if ($user->isAdmin()) {
            return $this->adminDashboard();
        }

        return view('dashboard.user', compact(
            'todayIncome',
            'todayExpense',
            'monthlyIncome',
            'monthlyExpense',
            'cashBalance',
            'bankAccounts',
            'digitalAccounts',
            'recentTransactions',
            'lowStockProducts',
            'chartData'
        ));
    }

    private function adminDashboard()
    {
        $totalUsers = User::where('role', 'user')->count();
        $totalTransactions = Transaction::count();
        $totalRevenue = Transaction::where('type', 'income')->sum('amount');
        $totalExpenses = Transaction::where('type', 'expense')->sum('amount');

        $recentUsers = User::where('role', 'user')
            ->latest()
            ->limit(10)
            ->get();

        $topUsers = User::withSum('transactions as total_income', 'amount')
            ->where('role', 'user')
            ->orderBy('total_income', 'desc')
            ->limit(5)
            ->get();

        return view('dashboard.admin', compact(
            'totalUsers',
            'totalTransactions',
            'totalRevenue',
            'totalExpenses',
            'recentUsers',
            'topUsers'
        ));
    }

    private function getChartData($user)
    {
        $dates = collect();
        $income = collect();
        $expense = collect();

        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $dates->push($date->format('M d'));

            $dayIncome = Transaction::where('user_id', $user->id)
                ->where('type', 'income')
                ->whereDate('date', $date)
                ->sum('amount');

            $dayExpense = Transaction::where('user_id', $user->id)
                ->where('type', 'expense')
                ->whereDate('date', $date)
                ->sum('amount');

            $income->push($dayIncome);
            $expense->push($dayExpense);
        }

        return [
            'labels' => $dates,
            'income' => $income,
            'expense' => $expense
        ];
    }
}
