<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Transaction;
use App\Models\Expense;
use App\Models\Invoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    public function dashboard()
    {
        $totalUsers = User::where('role', 'user')->count();
        $totalTransactions = Transaction::count();
        $totalRevenue = Transaction::where('type', 'income')->sum('amount');
        $totalExpenses = Transaction::where('type', 'expense')->sum('amount');

        $recentUsers = User::where('role', 'user')
            ->latest()
            ->limit(10)
            ->get();

        $topUsers = User::withSum('transactions as total_income', 'amount')
            ->where('role', 'user')
            ->orderBy('total_income', 'desc')
            ->limit(5)
            ->get();

        return view('dashboard.admin', compact(
            'totalUsers',
            'totalTransactions',
            'totalRevenue',
            'totalExpenses',
            'recentUsers',
            'topUsers'
        ));
    }

    public function users()
    {
        $users = User::where('role', 'user')
            ->withCount('transactions', 'invoices', 'expenses')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.users', compact('users'));
    }

    public function userDetails(User $user)
    {
        $user->loadCount('transactions', 'invoices', 'expenses');

        $recentTransactions = $user->transactions()
            ->with(['bankAccount', 'digitalPayment', 'expenseType'])
            ->orderBy('date', 'desc')
            ->limit(10)
            ->get();

        $totalIncome = $user->transactions()
            ->where('type', 'income')
            ->sum('amount');

        $totalExpense = $user->transactions()
            ->where('type', 'expense')
            ->sum('amount');

        return view('admin.user-details', compact('user', 'recentTransactions', 'totalIncome', 'totalExpense'));
    }

    public function reports()
    {
        // Get overall system statistics
        $totalUsers = User::where('role', 'user')->count();
        $totalRevenue = Transaction::where('type', 'income')->sum('amount');
        $totalExpenses = Transaction::where('type', 'expense')->sum('amount');
        $netProfit = $totalRevenue - $totalExpenses;

        // Monthly breakdown for current year
        $monthlyData = Transaction::select(
            DB::raw('MONTH(date) as month'),
            DB::raw('SUM(CASE WHEN type = "income" THEN amount ELSE 0 END) as income'),
            DB::raw('SUM(CASE WHEN type = "expense" THEN amount ELSE 0 END) as expense')
        )
            ->whereYear('date', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        // Top spending users
        $topSpendingUsers = User::withSum(['transactions as total_expense' => function ($query) {
            $query->where('type', 'expense');
        }], 'amount')
            ->where('role', 'user')
            ->orderBy('total_expense', 'desc')
            ->limit(10)
            ->get();

        return view('admin.reports', compact(
            'totalUsers',
            'totalRevenue',
            'totalExpenses',
            'netProfit',
            'monthlyData',
            'topSpendingUsers'
        ));
    }
}
